//
//  DataCenter.swift
//  Do It
//
//  Created by Jim Dovey on 10/13/19.
//  Copyright © 2019 Jim Dovey. All rights reserved.
//

import Foundation
import Combine
import SwiftUI

fileprivate let jsonDataName = "todoData.json"

extension UUID {
    static var null: UUID { UUID(uuid: UUID_NULL) }
}

fileprivate struct IndexSetIterator<C: Collection>: IteratorProtocol where C.Index == Int {
    private let source: C
    private var indices: IndexSet.Iterator
    
    init(_ source: C, _ indexes: IndexSet) {
        self.source = source
        self.indices = indexes.makeIterator()
    }
    
    mutating func next() -> C.Element? {
        guard let idx = indices.next() else { return nil }
        return source[idx]
    }
}

class DataCenter: ObservableObject {
    @Published var todoItems: [TodoItem] = []
    @Published var todoLists: [TodoItemList] = []
    @Published var defaultListID: UUID
    
    private var itemIndex: [UUID: Int] = [:]
    private var saveCancellable: AnyCancellable?
    
    var undoManager: UndoManager?
    
    let queue = DispatchQueue(label: "com.pragprog.swiftui.DataCenter.mutation")
    
    private init(_ todoData: TodoData) {
        self.todoItems = todoData.items
        self.todoLists = todoData.lists
        self.defaultListID = todoData.defaultListID
        queue.sync(execute: updateItemIndex)
        saveWhenChanged()
    }
    
    init(withTestItems todoData: TodoData) {
        self.todoItems = todoData.items
        self.todoLists = todoData.lists
        self.defaultListID = todoData.defaultListID
        self.undoManager = UndoManager()
        queue.sync(execute: updateItemIndex)
    }
    
    convenience init<S: Scheduler>(
        withTestItems todoData: TodoData,
        scheduler: S,
        subject: CurrentValueSubject<TodoData,Never>,
        subscribers: inout Set<AnyCancellable>
    ) {
        self.init(withTestItems: todoData)
        
        $todoLists
            .combineLatest($todoItems, $defaultListID, TodoData.init(lists:items:defaultListID:))
            .dropFirst()
            .debounce(for: .milliseconds(100), scheduler: scheduler)
            .subscribe(subject)
            .store(in: &subscribers)
    }

    convenience init() {
        let todoData: TodoData
        var scheduleItems = false
        do {
            todoData = try Self.loadJSON(from: Self.jsonDataURL)
        } catch {
            todoData = Self.createDefaultItems()
            scheduleItems = true
        }
        self.init(todoData)
        
        if scheduleItems {
            NotificationManager.shared.preflightFirstLaunchScheduling(for: self)
        }
    }

    private static func createDefaultItems() -> TodoData {
        let encoder = JSONEncoder()
        encoder.dateEncodingStrategy = .iso8601
        do {
            let data = try encoder.encode(defaultTodoData)
            writeData(data)
        } catch {
            fatalError("Failed to create default todo items! \(error)")
        }
        return defaultTodoData
    }

    private static let jsonDataURL: URL = {
        let baseURL: URL
        do {
            baseURL = try FileManager.default.url(
                for: .documentDirectory, in: .userDomainMask,
                appropriateFor: nil, create: true)
        } catch {
            let homeURL = URL(fileURLWithPath: NSHomeDirectory(),
                              isDirectory: true)
            baseURL = homeURL.appendingPathComponent("Documents")
        }
        return baseURL.appendingPathComponent(jsonDataName)
    }()
}

// MARK: Index Management

extension DataCenter {
    private func updateItemIndex() {
        dispatchPrecondition(condition: .onQueue(queue))
        
        itemIndex.removeAll(keepingCapacity: true)
        for (idx, item) in todoItems.enumerated() {
            itemIndex[item.id] = idx
        }
    }
}

// MARK: List/Item Lookups

extension DataCenter {
    @inlinable
    func list(for item: TodoItem) -> TodoItemList {
        queue.sync {
            guard let list = todoLists.first(where: { $0.id == item.listID }) else {
                preconditionFailure("No matching list for ID: \(item.listID)")
            }
            return list
        }
    }

    @inlinable
    func items<S: Sequence>(withIDs ids: S) -> [TodoItem] where S.Element == UUID {
        queue.sync {
            ids.compactMap { itemIndex[$0].map { todoItems[$0] } }
        }
    }
    
    @inlinable
    func list(withID identifier: UUID) -> TodoItemList? {
        queue.sync {
            todoLists.first { $0.id == identifier }
        }
    }
    
    @inlinable
    func item(withID identifier: UUID) -> TodoItem? {
        queue.sync {
            itemIndex[identifier].map { todoItems[$0] }
        }
    }
    
    @inlinable
    func items(in list: TodoItemList) -> [TodoItem] {
        items(withIDs: list.items)
    }
    
    var defaultItemList: TodoItemList {
        queue.sync {
            if defaultListID != UUID.null, let list = todoLists.first(where: {$0.id == defaultListID}) {
                return list
            }
            
            _fixDefaultListID()
            return todoLists[0]
        }
    }
    
    private func _fixDefaultListID() {
        dispatchPrecondition(condition: .onQueue(queue))
        
        // pick one or create one
        if todoLists.isEmpty {
            todoLists.append(TodoItemList(name: "Reminders",
                                          color: .purple,
                                          icon: "list.bullet"))
        }
        let list = todoLists.first!
        defaultListID = list.id
    }
    
    private var _defaultListIndex: Int {
        dispatchPrecondition(condition: .onQueue(queue))
        if defaultListID != UUID.null, let idx = todoLists.firstIndex(where: { $0.id == defaultListID }) {
            return idx
        }
        
        _fixDefaultListID()
        return 0
    }
}

// MARK: Undo/Redo Support

extension DataCenter {
    func runUndoableOperation<R>(_ perform: () throws -> R) rethrows -> R {
        dispatchPrecondition(condition: .onQueue(queue))
        
        // short-circuit if we're not recording undo actions
        guard let mgr = undoManager, mgr.isUndoRegistrationEnabled else {
            return try perform()
        }
        
        func applyItemDiff(_ diff: CollectionDifference<TodoItem>, to center: DataCenter) {
            guard let newItems = center.todoItems.applying(diff) else {
                return
            }
            center.todoItems = newItems
            center.updateItemIndex()
            if let mgr = center.undoManager {
                mgr.registerUndo(withTarget: center) { c in
                    c.queue.sync {
                        applyItemDiff(diff.inverse(), to: c)
                    }
                }
            }
        }
        
        func applyListDiff(_ diff: CollectionDifference<TodoItemList>,
                           to center: DataCenter) {
            guard let newLists = center.todoLists.applying(diff) else {
                return
            }
            center.todoLists = newLists
            if let mgr = center.undoManager {
                mgr.registerUndo(withTarget: center) { c in
                    c.queue.sync {
                        applyListDiff(diff.inverse(), to: c)
                    }
                }
            }
        }
        
        let listSnapshot = todoLists
        let itemSnapshot = todoItems
        
        let result = try perform()
        
        let listDiff = listSnapshot.difference(from: todoLists)
        let itemDiff = itemSnapshot.difference(from: todoItems)
        
        if listDiff.isEmpty && itemDiff.isEmpty {
            return result
        }
        
        registerUndo { center in
            center.queue.sync {
                center.undoManager?.beginUndoGrouping()
                if !itemDiff.isEmpty {
                    applyItemDiff(itemDiff, to: center)
                }
                if listDiff.count != 0 {
                    applyListDiff(listDiff, to: center)
                }
                center.undoManager?.endUndoGrouping()
            }
        }
        
        return result
    }
    
    func registerUndo(performing block: @escaping (DataCenter) -> Void) {
        guard let undoManager = undoManager, undoManager.isUndoRegistrationEnabled else {
            return
        }
        undoManager.beginUndoGrouping()
        undoManager.registerUndo(withTarget: self) { block($0) }
        undoManager.endUndoGrouping()
    }
}

// MARK: TodoItem Array Convenience

extension DataCenter {
    func addTodoItem(_ item: TodoItem, globalIndex: Int = .max) {
        queue.sync {
            _addTodoItem(item, globalIndex)
        }
    }
    
    private func _addTodoItem(_ item: TodoItem, _ index: Int = .max) {
        dispatchPrecondition(condition: .onQueue(queue))
        guard !self.todoItems.contains(where: { $0.id == item.id }) else { return }
        
        if let idx = todoLists.firstIndex(where: { $0.id == item.listID }) {
            if !todoLists[idx].items.contains(item.id) {
                todoLists[idx].items.append(item.id)
            }
            
            if index == .max {
                todoItems.append(item)
            }
            else {
                todoItems.insert(item, at: index)
            }
        }
        else {
            let idx = _defaultListIndex
            var updated = item
            updated.listID = todoLists[idx].id
            todoLists[idx].items.append(item.id)
            
            if index == .max {
                todoItems.append(updated)
            }
            else {
                todoItems.insert(updated, at: index)
            }
        }
        
        if index == .max {
            itemIndex[item.id] = todoItems.index(before: todoItems.endIndex)
        }
        else {
            itemIndex[item.id] = index
            for (id, idx) in itemIndex where idx > index {
                itemIndex[id] = idx + 1
            }
        }
        
        registerUndo {
            $0.removeTodoItems(withIDs: [item.id])
        }
    }
    
    private func _listIndex(for listID: UUID) -> Int? {
        dispatchPrecondition(condition: .onQueue(queue))
        return todoLists.firstIndex { $0.id == listID }
    }

    func removeTodoItems(atOffsets offsets: IndexSet, in list: TodoItemList) {
        queue.sync {
            runUndoableOperation {
                let ids = Set(IteratorSequence(IndexSetIterator(list.items, offsets)))
                if let idx = _listIndex(for: list.id) {
                    // inline the item removal, we know they're in a single list.
                    todoItems.removeAll { ids.contains($0.id) }
                    todoLists[idx].items.remove(atOffsets: offsets)
                    updateItemIndex()
                }
                else {
                    // use the generic algorithm
                    _removeTodoItems(ids)
                }
            }
        }
    }
    
    func removeTodoItems<S: Sequence>(withIDs ids: S) where S.Element == UUID {
        queue.sync {
            _removeTodoItems(Set(ids))
        }
    }
    
    func _removeTodoItems(_ ids: Set<UUID>) {
        dispatchPrecondition(condition: .onQueue(queue))
        runUndoableOperation {
            ids.forEach { itemIndex.removeValue(forKey: $0) }
            todoItems.removeAll { ids.contains($0.id) }
            for idx in todoLists.indices {
                todoLists[idx].items.removeAll { ids.contains($0) }
            }
        }
    }

    func updateTodoItem(_ item: TodoItem) {
        queue.sync {
            guard let idx = todoItems.firstIndex(where: { $0.id == item.id })
                else { _addTodoItem(item); return }
            
            if todoItems[idx].listID == item.listID {
                // nothing else needs to change
                let old = todoItems[idx]
                todoItems[idx] = item
                registerUndo { $0.updateTodoItem(old) }
            }
            else {
                runUndoableOperation {
                    // list membership changed!
                    if let listIdx = todoLists.firstIndex(where: { $0.id == todoItems[idx].listID }),
                        let subIdx = todoLists[listIdx].items.firstIndex(where: { $0 == todoItems[idx].id }) {
                        // remove from old list
                        todoLists[listIdx].items.remove(at: subIdx)
                    }
                    
                    if let listIdx = todoLists.firstIndex(where: { $0.id == item.listID }) {
                        todoLists[listIdx].items.append(item.id)
                    }
                    else {
                        // move to default list
                        let listIdx = _defaultListIndex
                        todoItems[idx].listID = defaultListID
                        todoLists[listIdx].items.append(item.id)
                    }
                }
            }
        }
    }

    func moveTodoItems(fromOffsets offsets: IndexSet, to index: Int, within list: TodoItemList) {
        queue.sync {
            guard let idx = _listIndex(for: list.id) else { return }
            let old = todoLists[idx].items
            todoLists[idx].items.move(fromOffsets: offsets, toOffset: index)
            
            registerUndo { center in
                guard let idx = center.todoLists.firstIndex(where: { $0.id == list.id }) else { return }
                center.queue.sync {
                    center.todoLists[idx].items = old
                }
                center.registerUndo {
                    $0.moveTodoItems(fromOffsets: offsets, to: index, within: list)
                }
            }
        }
    }
    
    private func _removeItemIDs(_ ids: Set<UUID>, fromListWithID listID: UUID) {
        dispatchPrecondition(condition: .onQueue(queue))
        guard let idx = todoLists.firstIndex(where: { $0.id == listID }) else {
            return
        }
        var newItems = todoLists[idx].items
        newItems.removeAll { ids.contains($0) }
        todoLists[idx].items = newItems
    }
    
    func moveTodoItems<C: Collection>(withIDs ids: C, toList list: TodoItemList, at index: Int) where C.Element == UUID {
        queue.sync {
            runUndoableOperation {
                guard let idx = _listIndex(for: list.id) else { return }
                let listItems = todoLists[idx].items
                let idSet = ids as? Set<UUID> ?? Set(ids)
                
                if !listItems.contains(where: { idSet.contains($0) }) {
                    // simple case--just an insert
                    var newItems = listItems
                    newItems.insert(contentsOf: ids, at: index)
                    todoLists[idx].items = newItems
                }
                else {
                    let _presentIndices = listItems.indices.compactMap {
                        idSet.contains(listItems[$0]) ? $0 : nil
                    }
                    let presentIndices = IndexSet(_presentIndices)
                    let countBeforeInsertionPoint = presentIndices.count(in: ..<index)
                    
                    // Now remove the pre-existing elements
                    var newItems = listItems
                    newItems.remove(atOffsets: presentIndices)
                    
                    // target index is adjusted downwards based on the number
                    // of items removed before the index
                    let targetIdx = index - countBeforeInsertionPoint
                    // insert moved item ids into the chosen location in the list,
                    // retaining the order in the input collection
                    newItems.insert(contentsOf: ids, at: targetIdx)
                    todoLists[idx].items = newItems
                }
                    
                // now update list IDs of any items moved
                
                for itemIdx in ids.compactMap({ itemIndex[$0] }) {
                    let listID = todoItems[itemIdx].listID
                    if listID != list.id {
                        _removeItemIDs(idSet, fromListWithID: todoItems[itemIdx].listID)
                        todoItems[itemIdx].listID = list.id
                    }
                }
            }
        }
    }

    func addList(_ list: TodoItemList) {
        queue.sync {
            _addList(list)
        }
    }
    
    private func _addList(_ list: TodoItemList) {
        dispatchPrecondition(condition: .onQueue(queue))
        
        todoLists.append(list)
        registerUndo {
            guard let idx = $0.todoLists.firstIndex(where: { $0.id == list.id }) else { return }
            $0.removeLists(atOffsets: IndexSet(integer: idx))
        }
    }

    func removeLists(atOffsets offsets: IndexSet) {
        queue.sync {
            let canUndo = undoManager != nil
                && undoManager!.isUndoRegistrationEnabled
            
            if canUndo {
                let removedLists = offsets.map { todoLists[$0] }
                let listIDs = removedLists.map { $0.id }
                todoLists.remove(atOffsets: offsets)
                
                let removedItems = removedLists.map { list in
                    todoItems.filter { $0.listID == list.id }
                }
                todoItems.removeAll { listIDs.contains($0.listID) }
                
                registerUndo {
                    $0.undoManager?.beginUndoGrouping()
                    for (num, list) in removedLists.enumerated() {
                        $0.addList(list)
                        for item in removedItems[num] {
                            $0.addTodoItem(item)
                        }
                    }
                    $0.undoManager?.endUndoGrouping()
                }
            }
            else {
                let listIDs = offsets.map { todoLists[$0].id }
                todoLists.remove(atOffsets: offsets)
                todoItems.removeAll { listIDs.contains($0.listID) }
            }
        }
    }

    func updateList(_ list: TodoItemList) {
        queue.sync {
            if let idx = todoLists.firstIndex(where: { $0.id == list.id }) {
                let old = todoLists[idx]
                todoLists[idx] = list
                registerUndo {
                    $0.updateList(old)
                }
            }
            else {
                _addList(list)
            }
        }
    }

    func moveLists(fromOffsets offsets: IndexSet, to index: Int) {
        queue.sync {
            let old = todoLists
            todoLists.move(fromOffsets: offsets, toOffset: index)
            let diff = old.difference(from: todoLists)
            registerUndo {
                guard let newLists = $0.todoLists.applying(diff) else { return }
                $0.todoLists = newLists
                $0.registerUndo {
                    $0.moveLists(fromOffsets: offsets, to: index)
                }
            }
        }
    }
}

// MARK: Save/Load Functionality

extension DataCenter {
    private func saveWhenChanged() {
        let encoder = JSONEncoder()
        encoder.dateEncodingStrategy = .iso8601

        let scheduler = UIBackgroundTaskScheduler(
            "Saving To-Do Items", target: DispatchQueue.global())

        self.saveCancellable = $todoLists
            // Combine list and items into a single TodoData object
            .combineLatest($todoItems, $defaultListID, TodoData.init(lists:items:defaultListID:))
            // Ignore the first value from this publisher; that's the initial
            // content that we started with, and we don't want to write that
            // out.
            .dropFirst()
            // Require there to be no changes for 100 ms before proceeding
            .debounce(for: .milliseconds(100), scheduler: DispatchQueue.global())
            // Wrap everything in a UIBackgroundTask
            .receive(on: scheduler)
            // Encode the TodoData using our JSONEncoder
            .encode(encoder: encoder)
            // When the encoder's output arrives, pipe it into the
            // writeData function to send it to disk.
            .sink(receiveCompletion: { _ in }, receiveValue: Self.writeData)
    }

    private static func writeData(_ data: Data) {
        do {
            try data.write(to: jsonDataURL,
                           options: [.completeFileProtection, .atomic])
        } catch {
            NSLog("Error writing JSON data: \(error)")
        }
    }

    private static func loadJSON<T: Decodable>(
        from url: URL,
        as type: T.Type = T.self
    ) throws -> T {
        let data = try Data(contentsOf: url,
                            options: [.mappedIfSafe])

        let decoder = JSONDecoder()
        decoder.dateDecodingStrategy = .iso8601
        return try decoder.decode(type, from: data)
    }
}
